#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "rng.h"
#include "api.h"
#include "gmp.h"
#include "kaz_api.h"
#include "sha256.h"

void HashMsg(const unsigned char *msg, unsigned long long mlen, unsigned char buf[32])
{
    sha256_t hash;
	sha256_init(&hash);
	sha256_update(&hash, msg, mlen);
	sha256_final(&hash, buf);
}

void KAZ_DS_PHI(mpz_t input, mpz_t output)
{
    mpz_t tmp, tmp1, tmp2;
    mpz_inits(tmp, tmp1, tmp2, NULL);

    mpz_set_ui(output, 1);

    int n=KAZ_DS_GET_PFactors(input);

    mpz_t *primefactor=malloc(n*sizeof(mpz_t));
    int *p=malloc(n*sizeof(int));
    int *e=malloc(n*sizeof(int));

    for(int i=0; i<n; i++) mpz_init(primefactor[i]);
    for(int i=0; i<n; i++) p[i]=0;
    for(int i=0; i<n; i++) e[i]=0;

    KAZ_DS_PFactors(input, primefactor, p, e);

    for(int i=0; i<n; i++){
        if(e[i]>1){
            mpz_set_ui(tmp1, p[i]);
            mpz_pow_ui(tmp, tmp1, e[i]);
            mpz_pow_ui(tmp2, tmp1, e[i]-1);
            mpz_sub(tmp, tmp, tmp2);
        }else{
            mpz_set_ui(tmp, p[i]-1);
        }

        mpz_mul(output, output, tmp);
    }

    mpz_clears(tmp, tmp1, tmp2, NULL);
}

void KAZ_DS_CRT(int size, mpz_t *x, mpz_t *modulus, mpz_t crt)
{
    mpz_t *c=malloc(size*sizeof(mpz_t));
    mpz_t u, prod;

    mpz_inits(u, prod, NULL);
    for(int i=0; i<size; i++) mpz_init(c[i]);

    mpz_set_ui(c[0], 0);

    for(int i=1; i<size; i++){
        mpz_set_ui(c[i], 1);

        for(int j=0; j<=i-1; j++){
            mpz_invert(u, modulus[j], modulus[i]);
            mpz_mul(c[i], c[i], u);
            mpz_mod(c[i], c[i], modulus[i]);
        }
    }

    mpz_set(u, x[0]);
    mpz_set(crt, u);

    for(int i=1; i<size; i++){
        mpz_sub(u, x[i], crt);
        mpz_mul(u, u, c[i]);
        mpz_mod(u, u, modulus[i]);
        mpz_set_ui(prod, 1);

        for(int j=0; j<=i-1; j++) mpz_mul(prod, prod, modulus[j]);

        mpz_mul(u, u, prod);
        mpz_add(crt, crt, u);
    }

    for(int i=0; i<size; i++) mpz_clear(c[i]);
    mpz_clears(u, prod, NULL);
}

void KAZ_DS_OrderBase(mpz_t Modular, mpz_t FiModular, mpz_t Base, mpz_t OrderBase)
{
    mpz_t tmp, t, a1;
    mpz_inits(tmp, t, a1, NULL);

    int nFiModular=KAZ_DS_GET_PFactors(FiModular);

    mpz_t *pFactors=malloc(nFiModular*sizeof(mpz_t));
    int *q=malloc(nFiModular*sizeof(int));
    int *e=malloc(nFiModular*sizeof(int));

    for(int i=0; i<nFiModular; i++) mpz_init(pFactors[i]);
    for(int i=0; i<nFiModular; i++) q[i]=0;
    for(int i=0; i<nFiModular; i++) e[i]=0;

    KAZ_DS_PFactors(FiModular, pFactors, q, e);

    mpz_set(t, FiModular);

    for(int i=0; i<nFiModular; i++){
        mpz_set_ui(tmp, q[i]);
        mpz_pow_ui(tmp, tmp, e[i]);
        mpz_divexact(t, t, tmp);

        mpz_powm(a1, Base, t, Modular);

        while(mpz_cmp_ui(a1, 1)!=0){
            mpz_powm_ui(a1, a1, q[i], Modular);
            mpz_mul_ui(t, t, q[i]);
            mpz_powm(a1, Base, t, Modular);
        }
    }

    mpz_set(OrderBase, t);
}

int KAZ_DS_GET_PFactors(mpz_t input)
{
    mpz_t inp, prod;
    mpz_inits(inp, prod, NULL);
    mpz_set(inp, input);
    mpz_set_ui(prod, 1);

    int div=2, count=0;
    int i=0;

    while(mpz_cmp_ui(inp, 1)>0){
        while(mpz_divisible_ui_p(inp, div)>0){
            count++;
            mpz_divexact_ui(inp, inp, div);
            mpz_mul_ui(prod, prod, div);
        }

        if(mpz_cmp_ui(prod, 1)>0){
            i++;
        }
        mpz_set_ui(prod, 1);
        div++;
        count=0;
    }

    return i;
}

void KAZ_DS_PFactors(mpz_t ord, mpz_t *pfacs, int *qlist, int *elist)
{
    mpz_t inp, prod;
    mpz_inits(inp, prod, NULL);
    mpz_set(inp, ord);
    mpz_set_ui(prod, 1);

    int div=2, count=0;
    unsigned long long i=0;

    while(mpz_cmp_ui(inp, 1)>0){
        while(mpz_divisible_ui_p(inp, div)>0){
            count++;
            mpz_divexact_ui(inp, inp, div);
            mpz_mul_ui(prod, prod, div);
        }

        if(mpz_cmp_ui(prod, 1)>0){

            mpz_set(pfacs[i], prod);
            qlist[i]=div;
            elist[i]=count;
            i++;
        }
        mpz_set_ui(prod, 1);
        div++;
        count=0;
    }

    mpz_clear(inp);
}

void KAZ_DS_RANDOM(int lb, int ub, mpz_t out)
{
	mpz_t lbound, ubound;

	gmp_randstate_t gmpRandState;
    gmp_randinit_mt(gmpRandState);
	mpz_inits(lbound, ubound, NULL);

	mpz_ui_pow_ui(lbound, 2, lb);
	mpz_ui_pow_ui(ubound, 2, ub);

	unsigned int sd=100000;

	do{
		gmp_randseed_ui(gmpRandState, rand()+sd);
		mpz_urandomb(out, gmpRandState, ub);
		sd+=1;
	}while((mpz_cmp(out, lbound) == -1) || (mpz_cmp(out, ubound) == 1));
}

void KAZ_DS_FILTER(mpz_t vq, mpz_t grg, mpz_t q, mpz_t h, mpz_t wb, mpz_t w17)
{
    mpz_t modulus, GRgQ, GCD, soln, VQ, check1, check2;

    mpz_inits(modulus, GRgQ, GCD, soln, VQ, check1, check2, NULL);

    mpz_mul(GRgQ, grg, q);

    int nGRgQ=KAZ_DS_GET_PFactors(GRgQ);

    mpz_t *pFactors=malloc(nGRgQ*sizeof(mpz_t));
    int *p=malloc(nGRgQ*sizeof(int));
    int *e=malloc(nGRgQ*sizeof(int));

    for(int i=0; i<nGRgQ; i++) mpz_init(pFactors[i]);
    for(int i=0; i<nGRgQ; i++) p[i]=0;
    for(int i=0; i<nGRgQ; i++) e[i]=0;

    KAZ_DS_PFactors(GRgQ, pFactors, p, e);

    mpz_t *x=malloc(2*sizeof(mpz_t));
    mpz_t *y=malloc(2*sizeof(mpz_t));

    for(int i=0; i<2; i++) mpz_init(x[i]);
    for(int i=0; i<2; i++) mpz_init(y[i]);

    mpz_set_ui(w17, 0);
    mpz_set_ui(modulus, 1);

    for(int i=0; i<nGRgQ; i++){
        mpz_set_ui(soln, 0);
        while(mpz_cmp(soln, pFactors[i])<0){
            mpz_gcd(GCD, q, pFactors[i]);
            mpz_mod(check1, soln, GCD);
            mpz_mod(check2, h, GCD);
            if(mpz_cmp(check1, check2)!=0){
                mpz_add_ui(soln, soln, 1);
                continue;
            }

            mpz_gcd(GCD, grg, pFactors[i]);
            mpz_mod(check1, soln, GCD);
            mpz_mul(check2, h, vq);
            mpz_mod(check2, check2, GCD);
            if(mpz_cmp(check1, check2)!=0){
                mpz_add_ui(soln, soln, 1);
                continue;
            }

            mpz_gcd(GCD, GRgQ, pFactors[i]);
            mpz_powm(check1, soln, wb, GCD);
            mpz_powm(check2, h, wb, GCD);
            if(mpz_cmp(check1, check2)!=0){
                mpz_add_ui(soln, soln, 1);
                continue;
            }

            break;
        }

        mpz_set(x[0], w17);
        mpz_set(x[1], soln);

        mpz_set(y[0], modulus);
        mpz_set(y[1], pFactors[i]);

        KAZ_DS_CRT(2, x, y, w17);
        mpz_mul(modulus, modulus, pFactors[i]);
    }

    mpz_clears(modulus, GRgQ, GCD, soln, VQ, check1, check2, NULL);
}

void KAZ_DS_KeyGen(unsigned char *kaz_ds_verify_key, unsigned char *kaz_ds_sign_key)
{
    mpz_t N, phiGRg, GRg, q, Q, phiQ, phiGRgQ;
    mpz_t ALPHA, GRgq, V, aF, GRgQ, GV0, GV1, GVGRgQ, GaFGRgQ, Z1, Z2, GZ1GRgQ, GZ2GRgQ;
    mpz_t CHECK1, CHECK2, CHECK3, W0, WA, WB, GALPHAGRgQ, tmp, rnd;

    mpz_inits(N, phiGRg, GRg, q, Q, phiQ, phiGRgQ, NULL);
    mpz_inits(ALPHA, GRgq, V, aF, GRgQ, GV0, GV1, GVGRgQ, GaFGRgQ, Z1, Z2, GZ1GRgQ, GZ2GRgQ, NULL);
    mpz_inits(CHECK1, CHECK2, CHECK3, W0, WA, WB, GALPHAGRgQ, tmp, rnd, NULL);

    //1) Get all system parameters
    mpz_set_str(N, KAZ_DS_SP_N, 10);
    mpz_set_str(GRg, KAZ_DS_SP_GRg, 10);
    mpz_set_str(phiGRg, KAZ_DS_SP_PHIGRg, 10);
    mpz_set_str(q, KAZ_DS_SP_q, 10);
    mpz_set_str(Q, KAZ_DS_SP_Q, 10);
    mpz_set_str(phiQ, KAZ_DS_SP_PHIQ, 10);
    mpz_set_str(phiGRgQ, KAZ_DS_SP_PHIGRgQ, 10);

    int n=KAZ_DS_SP_n;
    int nphiGg=KAZ_DS_SP_nPHIGg;
    int lenGV=0;
    int lenGALPHA=0;
    int DEFAULT_LEN=110;

    do{
        do{
            // Generate ALPHA
            KAZ_DS_RANDOM(nphiGg-2, nphiGg-1, ALPHA);
            mpz_nextprime(ALPHA, ALPHA);

            // Generate V
            mpz_mul(GRgq, GRg, q);
            mpz_mod(V, ALPHA, GRgq);
            mpz_mod(aF, V, GRg);

            mpz_mul(GRgQ, GRg, Q);

            mpz_gcd(GV0, V, GRgQ);
            mpz_gcd(GV1, aF, GRgQ);
        }while(mpz_cmp_ui(GV0, 1)!=0 || mpz_cmp_ui(GV1, 1)!=0);

        // Order V in Z_{GRgQ} as GVGRgQ
        KAZ_DS_OrderBase(GRgQ, phiGRgQ, V, GVGRgQ);

        // Compute WA
        mpz_gcd(W0, phiGRg, GVGRgQ);
        mpz_divexact(WA, GVGRgQ, W0);

        // Compute Z1
        mpz_invert(Z1, V, GRgQ);
        mpz_mul(Z1, Z1, ALPHA);
        mpz_mod(Z1, Z1, GRgQ);

        // Compute Order Z1 in Z_{GRgQ} as GZ1GRgQ
        KAZ_DS_OrderBase(GRgQ, phiGRgQ, Z1, GZ1GRgQ);

        // Compute Z2
        mpz_invert(Z2, aF, GRgQ);
        mpz_mul(Z2, Z2, ALPHA);
        mpz_mod(Z2, Z2, GRgQ);

        // Compute Order Z2 in Z_{GRgQ} as GZ2GRgQ
        KAZ_DS_OrderBase(GRgQ, phiGRgQ, Z2, GZ2GRgQ);

        // Order aF in Z_{GRgQ} as GaFGRgQ
        KAZ_DS_OrderBase(GRgQ, phiGRgQ, aF, GaFGRgQ);

        mpz_mul(CHECK1, phiQ, WA);
        mpz_mod(CHECK1, CHECK1, GZ1GRgQ);
        mpz_mul(CHECK2, phiQ, WA);
        mpz_mod(CHECK2, CHECK2, GZ2GRgQ);
        mpz_mod(CHECK3, GVGRgQ, GaFGRgQ);

        // Order ALPHA in Z_{GRgQ} as GALPHAGRgQ
        KAZ_DS_OrderBase(GRgQ, phiGRgQ, ALPHA, GALPHAGRgQ);

        lenGV=mpz_sizeinbase(GVGRgQ, 2);
        lenGALPHA=mpz_sizeinbase(GALPHAGRgQ, 2);
    }while(mpz_cmp_ui(CHECK1, 0)==0 || mpz_cmp_ui(CHECK2, 0)==0 || mpz_cmp_ui(CHECK3, 0)!=0 ||
           lenGV<DEFAULT_LEN || lenGALPHA<DEFAULT_LEN);

    mpz_gcd(tmp, GALPHAGRgQ, phiQ);
    mpz_divexact(WB, GALPHAGRgQ, tmp);

    //6) Set kaz_ds_sign_key=(ALPHA) & kaz_ds_verify_key=(V, W)
    size_t ALPHASIZE=mpz_sizeinbase(ALPHA, 16);
	size_t VSIZE=mpz_sizeinbase(V, 16);
	size_t WASIZE=mpz_sizeinbase(WA, 16);
	size_t WBSIZE=mpz_sizeinbase(WB, 16);

	unsigned char *ALPHABYTE=(unsigned char*) malloc(ALPHASIZE*sizeof(unsigned char));
	mpz_export(ALPHABYTE, &ALPHASIZE, 1, sizeof(char), 0, 0, ALPHA);

	unsigned char *VBYTE=(unsigned char*) malloc(VSIZE*sizeof(unsigned char));
	mpz_export(VBYTE, &VSIZE, 1, sizeof(char), 0, 0, V);

	unsigned char *WABYTE=(unsigned char*) malloc(WASIZE*sizeof(unsigned char));
	mpz_export(WABYTE, &WASIZE, 1, sizeof(char), 0, 0, WA);

	unsigned char *WBBYTE=(unsigned char*) malloc(WBSIZE*sizeof(unsigned char));
	mpz_export(WBBYTE, &WBSIZE, 1, sizeof(char), 0, 0, WB);

	for(int i=0; i<CRYPTO_SECRETKEYBYTES; i++) kaz_ds_sign_key[i]=0;

	int je=CRYPTO_SECRETKEYBYTES-1;
	for(int i=WBSIZE-1; i>=0; i--){
		kaz_ds_sign_key[je]=WBBYTE[i];
		je--;
	}

	je=CRYPTO_SECRETKEYBYTES-KAZ_DS_WBBYTES-1;
	for(int i=VSIZE-1; i>=0; i--){
		kaz_ds_sign_key[je]=VBYTE[i];
		je--;
	}

	je=CRYPTO_SECRETKEYBYTES-KAZ_DS_WBBYTES-KAZ_DS_VBYTES-1;
	for(int i=ALPHASIZE-1; i>=0; i--){
		kaz_ds_sign_key[je]=ALPHABYTE[i];
		je--;
	}

	for(int i=0; i<CRYPTO_PUBLICKEYBYTES; i++) kaz_ds_verify_key[i]=0;

	je=CRYPTO_PUBLICKEYBYTES-1;
	for(int i=WBSIZE-1; i>=0; i--){
		kaz_ds_verify_key[je]=WBBYTE[i];
		je--;
	}

	je=CRYPTO_PUBLICKEYBYTES-KAZ_DS_WBBYTES-1;
	for(int i=WASIZE-1; i>=0; i--){
		kaz_ds_verify_key[je]=WABYTE[i];
		je--;
	}

	je=CRYPTO_PUBLICKEYBYTES-KAZ_DS_WBBYTES-KAZ_DS_WABYTES-1;
	for(int i=VSIZE-1; i>=0; i--){
		kaz_ds_verify_key[je]=VBYTE[i];
		je--;
	}

	//mpz_gcd(tmp, V, GRgQ);gmp_printf("GCD(V,GRgQ)=%Zd\n", tmp);
    //mpz_gcd(tmp, aF, GRgQ);gmp_printf("GCD(aF,GRgQ)=%Zd\n", tmp);
    //mpz_mul(tmp, phiQ, W);mpz_mod(tmp, tmp, GZ1GRgQ);gmp_printf("phiQW mod GZ1GRgQ=%Zd\n", tmp);
    //mpz_mul(tmp, phiQ, W);mpz_mod(tmp, tmp, GZ2GRgQ);gmp_printf("phiQW mod GZ2GRgQ=%Zd\n", tmp);
    //gmp_printf("GZ1GRgQ=%Zd\n", GZ1GRgQ);
    //gmp_printf("GZ2GRgQ=%Zd\n", GZ2GRgQ);
	//gmp_printf("GV0=%Zd\nGV1=%Zd\nCHECK1=%Zd\nCHECK2=%Zd\n\n", GV0, GV1, GZ1GRgQ, GZ2GRgQ);
    //gmp_printf("ALPHA=%Zd\nV=%Zd\nWA=%Zd\nWB=%Zd\n", ALPHA, V, WA, WB);
	mpz_clears(N, phiGRg, GRg, q, Q, phiQ, phiGRgQ, NULL);
    mpz_clears(ALPHA, GRgq, V, aF, GRgQ, GV0, GV1, GVGRgQ, GaFGRgQ, Z1, Z2, GZ1GRgQ, GZ2GRgQ, NULL);
    mpz_clears(CHECK1, CHECK2, CHECK3, W0, WA, WB, GALPHAGRgQ, tmp, rnd, NULL);
}

int KAZ_DS_SIGNATURE(unsigned char *signature, unsigned long long *signlen, const unsigned char *m, unsigned long long mlen, const unsigned char *sk)
{
    mpz_t GRg, phiGRg, phiphiGRg, Q, phiQ, phiGRgQ, GRgqQ, phiGRgqQ, ALPHA, V, WB;
    mpz_t hashValue, saltValue, GhQ, phiGhQ, GRgQ, GhGRgQ, GVGRgQ, BETA, GBETAGhQ, GALPHAGRgQ;
    mpz_t phiGhGRgQWB, GBETAGhQinvWB, CHECK1, CHECK2, CHECK3, CHECK4, CHECK5, CHECK6, r0, r1, S, tmp, tmp2, rnd;

    mpz_inits(GRg, phiGRg, phiphiGRg, Q, phiQ, phiGRgQ, GRgqQ, phiGRgqQ, ALPHA, V, WB, NULL);
    mpz_inits(hashValue, saltValue, GhQ, phiGhQ, GRgQ, GhGRgQ, GVGRgQ, BETA, GBETAGhQ, GALPHAGRgQ, NULL);
    mpz_inits(phiGhGRgQWB, GBETAGhQinvWB, CHECK1, CHECK2, CHECK3, CHECK4, CHECK5, CHECK6, r0, r1, S, tmp, tmp2, rnd, NULL);

    //1) Get all system parameters
    mpz_set_str(GRg, KAZ_DS_SP_GRg, 10);
    mpz_set_str(phiGRg, KAZ_DS_SP_PHIGRg, 10);
    mpz_set_str(phiphiGRg, KAZ_DS_SP_PHIPHIGRg, 10);
    mpz_set_str(Q, KAZ_DS_SP_Q, 10);
    mpz_set_str(phiQ, KAZ_DS_SP_PHIQ, 10);
    mpz_set_str(phiGRgQ, KAZ_DS_SP_PHIGRgQ, 10);
    mpz_set_str(GRgqQ, KAZ_DS_SP_GRgqQ, 10);
    mpz_set_str(phiGRgqQ, KAZ_DS_SP_PHIGRgqQ, 10);

    int n=KAZ_DS_SP_n;
    int nphiGg=KAZ_DS_SP_nPHIGg;

    //2) Get kaz_ds_sign_key=(ALPHA, V, WB)
	unsigned char *ALPHABYTE=(unsigned char*) malloc((KAZ_DS_ALPHABYTES)*sizeof(unsigned char));
	unsigned char *VBYTE=(unsigned char*) malloc((KAZ_DS_VBYTES)*sizeof(unsigned char));
	unsigned char *WBBYTE=(unsigned char*) malloc((KAZ_DS_WBBYTES)*sizeof(unsigned char));

	for(int i=0; i<KAZ_DS_ALPHABYTES; i++) ALPHABYTE[i]=0;
	for(int i=0; i<KAZ_DS_VBYTES; i++) VBYTE[i]=0;
	for(int i=0; i<KAZ_DS_WBBYTES; i++) WBBYTE[i]=0;

	for(int i=0; i<KAZ_DS_ALPHABYTES; i++){ALPHABYTE[i]=sk[i];}
	for(int i=0; i<KAZ_DS_VBYTES; i++){VBYTE[i]=sk[i+KAZ_DS_ALPHABYTES];}
	for(int i=0; i<KAZ_DS_WBBYTES; i++){WBBYTE[i]=sk[i+KAZ_DS_ALPHABYTES+KAZ_DS_VBYTES];}

	mpz_import(ALPHA, KAZ_DS_ALPHABYTES, 1, sizeof(char), 0, 0, ALPHABYTE);
	mpz_import(V, KAZ_DS_VBYTES, 1, sizeof(char), 0, 0, VBYTE);
	mpz_import(WB, KAZ_DS_WBBYTES, 1, sizeof(char), 0, 0, WBBYTE);

	unsigned long long saltsize=32;
	unsigned char buf[CRYPTO_BYTES]={0};
	unsigned char *msgsalt;
	unsigned char *salt;

    do{
        msgsalt=(unsigned char*) calloc(mlen+saltsize, sizeof(unsigned char));
        salt=(unsigned char*) calloc(saltsize, sizeof(unsigned char));

        randombytes(salt, saltsize);

        for(int i=0; i<mlen+saltsize; i++) msgsalt[i]=0;
        for(int i=0; i<mlen; i++) msgsalt[i]=m[i];
        for(int i=0; i<saltsize; i++) msgsalt[i+mlen]=salt[i];

        HashMsg(msgsalt, mlen+saltsize, buf);

        mpz_import(hashValue, CRYPTO_BYTES, 1, sizeof(char), 0, 0, buf);
        mpz_import(saltValue, saltsize, 1, sizeof(char), 0, 0, salt);

        mpz_nextprime(hashValue, hashValue);

        // Compute Order hashValue in Z_{Q} as GhQ
        KAZ_DS_OrderBase(Q, phiQ, hashValue, GhQ);

        // Compute Order hashValue in Z_{GRgQ} as GhGRgQ
        mpz_mul(GRgQ, GRg, Q);
        KAZ_DS_OrderBase(GRgQ, phiGRgQ, hashValue, GhGRgQ);

        // Order V in Z_{GRgQ} as GVGRgQ
        KAZ_DS_OrderBase(GRgQ, phiGRgQ, V, GVGRgQ);

        // Order ALPHA in Z_{GRgQ} as GALPHAGRgQ
        KAZ_DS_OrderBase(GRgQ, phiGRgQ, ALPHA, GALPHAGRgQ);

        // Checking GALPHAGRgQ=0 mod GhGRgQ
        mpz_mod(CHECK1, GALPHAGRgQ, GhGRgQ);
        // Checking GVGRgQ=0 mod GhGRgQ
        mpz_mod(CHECK2, GVGRgQ, GhGRgQ);
        // Checking GhGRgQ=0 mod WB
        mpz_mod(CHECK3, GhGRgQ, WB);
        // Checking phi(Q)=0 mod GhQ
        mpz_mod(CHECK4, phiQ, GhQ);
    }while(mpz_cmp_ui(CHECK1, 0)!=0 || mpz_cmp_ui(CHECK2, 0)!=0 || mpz_cmp_ui(CHECK3, 0)!=0 || mpz_cmp_ui(CHECK4, 0)!=0 ||
              mpz_cmp(GVGRgQ, GhGRgQ)<0 || mpz_cmp(GALPHAGRgQ, GhGRgQ)<0);

    do{
        // Choose random ephemeral prime
        KAZ_DS_RANDOM(nphiGg-2, nphiGg-1, BETA);
        mpz_nextprime(BETA, BETA);

        // Compute Order BETA in Z_{GhQ} as GBETAGhQ
        KAZ_DS_PHI(GhQ, phiGhQ);
        KAZ_DS_OrderBase(GhQ, phiGhQ, BETA, GBETAGhQ);

        // Compute Order BETA in Z_{GhGRgQ/WB} as GBETAGhQinvWB
        mpz_divexact(tmp, GhGRgQ, WB);
        KAZ_DS_PHI(tmp, phiGhGRgQWB);
        KAZ_DS_OrderBase(tmp, phiGhGRgQWB, BETA, GBETAGhQinvWB);

        // Checking phi(phi(GRg))=0 mod GBETAGhQ
        mpz_mod(CHECK5, phiphiGRg, GBETAGhQ);
        // Checking phi(phi(GRg))=0 mod GBETAGhQinvWB
        mpz_mod(CHECK6, phiphiGRg, GBETAGhQinvWB);
        //gmp_printf("%Zd %Zd %Zd %Zd %Zd %Zd\n", CHECK1, CHECK2, CHECK3, CHECK4, CHECK5, CHECK6);
    }while(mpz_cmp_ui(CHECK5, 0)!=0 || mpz_cmp_ui(CHECK6, 0)!=0);

    KAZ_DS_RANDOM(255, 256, r0);
    KAZ_DS_RANDOM(255, 256, r1);

    //4) Compute Signature
    mpz_mul(tmp, phiGRgQ, r0);
    mpz_add(tmp, tmp, phiQ);
    mpz_powm(S, ALPHA, tmp, GRgqQ);

    mpz_powm(tmp, BETA, phiphiGRg, phiGRgqQ);
    mpz_mul(tmp2, phiGRgQ, r1);
    mpz_add(tmp, tmp, tmp2);
    mpz_powm(tmp2, hashValue, tmp, GRgqQ);

    mpz_mul(S, S, tmp2);
    mpz_mod(S, S, GRgqQ);

    //5) Set signature=(S, SALT)
    size_t SSIZE=mpz_sizeinbase(S, 16);

    for(int i=0; i<mlen+KAZ_DS_SBYTES+saltsize; i++) signature[i]=0;

	unsigned char *SBYTE=(unsigned char*) malloc(SSIZE*sizeof(unsigned char));
	mpz_export(SBYTE, &SSIZE, 1, sizeof(char), 0, 0, S);

	int je=mlen+saltsize+KAZ_DS_SBYTES-1;
	for(int i=mlen-1; i>=0; i--){
		signature[je]=m[i];
		je--;
	}

	je=saltsize+KAZ_DS_SBYTES-1;
	for(int i=saltsize-1; i>=0; i--){
		signature[je]=salt[i];
		je--;
	}

	je=KAZ_DS_SBYTES-1;
	for(int i=SSIZE-1; i>=0; i--){
		signature[je]=SBYTE[i];
		je--;
	}

	*signlen=mlen+saltsize+KAZ_DS_SBYTES;

    free(SBYTE);
    free(msgsalt);
    free(salt);

    //gmp_printf("phiGRgQ=%Zd\nphiQ=%Zd\nphiphiGRg=%Zd\nphiGRgqQ=%Zd\n", phiGRgQ, phiQ, phiphiGRg, phiGRgqQ);
    //gmp_printf("ALPHA=%Zd\nV=%Zd\nWB=%Zd\nh=%Zd\nsalt=%Zd\nS=%Zd\n", ALPHA, V, WB, hashValue, saltValue, S);
    //gmp_printf("h=%Zd\nsalt=%Zd\nS=%Zd\n", hashValue, saltValue, S);

	mpz_inits(GRg, phiGRg, phiphiGRg, Q, phiQ, phiGRgQ, GRgqQ, phiGRgqQ, ALPHA, V, WB, NULL);
    mpz_inits(hashValue, saltValue, GhQ, phiGhQ, GRgQ, GhGRgQ, GVGRgQ, BETA, GBETAGhQ, GALPHAGRgQ, NULL);
    mpz_inits(phiGhGRgQWB, GBETAGhQinvWB, CHECK1, CHECK2, CHECK3, CHECK4, CHECK5, CHECK6, r0, r1, S, tmp, tmp2, rnd, NULL);

    return 0;
}

int KAZ_DS_VERIFICATION(unsigned char *m, unsigned long long *mlen, const unsigned char *sm, unsigned long long smlen, const unsigned char *pk)
{
    mpz_t N, g, Gg, R, GRg, q, Q, phiQ, GRgqQ, hashValue, V, WA, WB, S, saltValue;
    mpz_t GRgQ, phiGRgQ, GhGRgQ, GVGRgQ, aF, W0, W1, W2, W3, W4, W5, W6, W7, W8, W9, W10;
    mpz_t W11, W12, W13, W14, W15, W16, W17, W18, VQ, aFQ, y1, y2, rnd, tmp;

    mpz_inits(N, g, Gg, R, GRg, q, Q, phiQ, GRgqQ, hashValue, V, WA, WB, S, saltValue, NULL);
    mpz_inits(GRgQ, phiGRgQ, GhGRgQ, GVGRgQ, aF, W0, W1, W2, W3, W4, W5, W6, W7, W8, W9, W10, NULL);
    mpz_inits(W11, W12, W13, W14, W15, W16, W17, W18, VQ, aFQ, y1, y2, rnd, tmp, NULL);

    //1) Get all system parameters
    mpz_set_str(N, KAZ_DS_SP_N, 10);
    mpz_set_str(g, KAZ_DS_SP_G, 10);
    mpz_set_str(Gg, KAZ_DS_SP_Gg, 10);
    mpz_set_str(R, KAZ_DS_SP_R, 10);
    mpz_set_str(GRg, KAZ_DS_SP_GRg, 10);
    mpz_set_str(q, KAZ_DS_SP_q, 10);
    mpz_set_str(Q, KAZ_DS_SP_Q, 10);
    mpz_set_str(phiQ, KAZ_DS_SP_PHIQ, 10);
    mpz_set_str(phiGRgQ, KAZ_DS_SP_PHIGRgQ, 10);
    mpz_set_str(GRgqQ, KAZ_DS_SP_GRgqQ, 10);

    int n=KAZ_DS_SP_n;

    //2) Get kaz_ds_verify_key=(V, W)
	unsigned char *VBYTE=(unsigned char*) malloc((KAZ_DS_VBYTES)*sizeof(unsigned char));
	unsigned char *WABYTE=(unsigned char*) malloc((KAZ_DS_WABYTES)*sizeof(unsigned char));
	unsigned char *WBBYTE=(unsigned char*) malloc((KAZ_DS_WBBYTES)*sizeof(unsigned char));

	for(int i=0; i<KAZ_DS_VBYTES; i++) VBYTE[i]=0;
	for(int i=0; i<KAZ_DS_WABYTES; i++) WABYTE[i]=0;
	for(int i=0; i<KAZ_DS_WBBYTES; i++) WBBYTE[i]=0;

	for(int i=0; i<KAZ_DS_VBYTES; i++){VBYTE[i]=pk[i];}
	for(int i=0; i<KAZ_DS_WABYTES; i++){WABYTE[i]=pk[i+KAZ_DS_VBYTES];}
	for(int i=0; i<KAZ_DS_WBBYTES; i++){WBBYTE[i]=pk[i+KAZ_DS_VBYTES+KAZ_DS_WABYTES];}

	mpz_import(V, KAZ_DS_VBYTES, 1, sizeof(char), 0, 0, VBYTE);
	mpz_import(WA, KAZ_DS_WABYTES, 1, sizeof(char), 0, 0, WABYTE);
	mpz_import(WB, KAZ_DS_WBBYTES, 1, sizeof(char), 0, 0, WBBYTE);

    //3) Get signature=(S, SALT, m)
    int len=smlen-32-KAZ_DS_SBYTES;

    unsigned char *SBYTE=(unsigned char*) malloc(KAZ_DS_SBYTES*sizeof(unsigned char));
	unsigned char *SALTBYTE=(unsigned char*) malloc(32*sizeof(unsigned char));
	unsigned char *MBYTE=(unsigned char*) malloc(len*sizeof(unsigned char));

	for(int i=0; i<KAZ_DS_SBYTES; i++) SBYTE[i]=0;
	for(int i=0; i<32; i++) SALTBYTE[i]=0;
	for(int i=0; i<len; i++) MBYTE[i]=0;

	for(int i=0; i<KAZ_DS_SBYTES; i++){SBYTE[i]=sm[i];}
	for(int i=0; i<32; i++){SALTBYTE[i]=sm[i+KAZ_DS_SBYTES];}
	for(int i=0; i<len; i++){MBYTE[i]=sm[i+KAZ_DS_SBYTES+32];}

    mpz_import(S, KAZ_DS_SBYTES, 1, sizeof(char), 0, 0, SBYTE);
	mpz_import(saltValue, 32, 1, sizeof(char), 0, 0, SALTBYTE);

	unsigned char *msgsalt=(unsigned char*) malloc((len+32)*sizeof(unsigned char));

    for(int i=0; i<len+32; i++) msgsalt[i]=0;
    for(int i=0; i<len; i++) msgsalt[i]=MBYTE[i];
    for(int i=0; i<32; i++) msgsalt[i+len]=SALTBYTE[i];

	//4) Compute the hash value of the message
    unsigned char buf[CRYPTO_BYTES]={0};
    HashMsg(msgsalt, len+32, buf);

    //gmp_printf("h=%Zd\nsalt=%Zd\nS=%Zd\n", hashValue, saltValue, S);

    mpz_import(hashValue, CRYPTO_BYTES, 1, sizeof(char), 0, 0, buf);
    mpz_nextprime(hashValue, hashValue);

    //gmp_printf("h=%Zd\nsalt=%Zd\nS=%Zd\n", hashValue, saltValue, S);

    // Compute Order hashValue in Z_{GRgQ} as GhGRgQ
    mpz_mul(GRgQ, GRg, Q);
    KAZ_DS_OrderBase(GRgQ, phiGRgQ, hashValue, GhGRgQ);

    // Order V in Z_{GRgQ} as GVGRgQ
    KAZ_DS_OrderBase(GRgQ, phiGRgQ, V, GVGRgQ);

    //5) Filtering Procedures
    //FILTER 1
    // Checking GVGRgQ=0 mod GhGRgQ
    mpz_mod(tmp, GVGRgQ, GhGRgQ);

    if(mpz_cmp_ui(tmp, 0)!=0){
        printf("Filter 1...\n");
        return -4;
    }

    //FILTER 2
    mpz_mod(W0, S, GRgqQ);
    mpz_sub(W0, W0, S);

    if(mpz_cmp_ui(W0, 0)!=0){
        printf("Filter 2...\n");
        return -4;
    }

    //FILTER 3
    mpz_invert(W1, hashValue, GRgQ);
    mpz_mul(W1, W1, S);
    mpz_mod(W1, W1, GRgQ); // W1=S/h mod GRgQ
    mpz_powm(W2, V, phiQ, GRgQ); // W2=V^(phiQ) mod GRgQ
    mpz_sub(W3, W1, W2);

    if(mpz_cmp_ui(W3, 0)==0){
        printf("Filter 3...\n");
        return -4;
    }

    //FILTER 4
    mpz_mod(aF, V, GRg);
    mpz_invert(W4, hashValue, GRgQ);
    mpz_mul(W4, W4, S);
    mpz_mod(W4, W4, GRgQ);
    mpz_powm(W5, aF, phiQ, GRgQ);
    mpz_sub(W6, W4, W5);

    if(mpz_cmp_ui(W6, 0)==0){
        printf("Filter 4...\n");
        return -4;
    }

    //FILTER 5
    mpz_invert(W7, hashValue, GRgQ);
    mpz_mul(W7, W7, S);
    mpz_mod(W7, W7, GRgQ);
    mpz_invert(W8, W2, GRgQ);
    mpz_mul(W8, W7, W8);
    mpz_mod(W8, W8, GRgQ);
    mpz_powm(W9, W8, WA, GRgQ);

    if(mpz_cmp_ui(W9, 1)==0){
        printf("Filter 5...\n");
        return -4;
    }

    //FILTER 6
    mpz_invert(W10, hashValue, GRgQ);
    mpz_mul(W10, W10, S);
    mpz_mod(W10, W10, GRgQ);
    mpz_invert(W11, W5, GRgQ);
    mpz_mul(W11, W10, W11);
    mpz_mod(W11, W11, GRgQ);
    mpz_powm(W12, W11, WA, GRgQ);

    if(mpz_cmp_ui(W12, 1)==0){
        printf("Filter 6...\n");
        return -4;
    }

    //FILTER 7
    mpz_invert(W13, hashValue, Q);
    mpz_mul(W13, W13, S);
    mpz_mod(W13, W13, Q);

    if(mpz_cmp_ui(W13, 1)!=0){
        printf("Filter 7...\n");
        return -4;
    }

    //FILTER 8
    mpz_powm(W14, S, WB, GRgQ);
    mpz_powm(W15, hashValue, WB, GRgQ);
    mpz_sub(W16, W14, W15);

    if(mpz_cmp_ui(W16, 0)!=0){
        printf("Filter 8...\n");
        return -4;
    }

    //FILTER 9
    mpz_powm(VQ, V, phiQ, GRg);
    KAZ_DS_FILTER(VQ, GRg, Q, hashValue, WB, W17);
    mpz_sub(W18, W17, S);
    mpz_mod(W18, W18, GRgQ);

    if(mpz_cmp_ui(W18, 0)==0){
        printf("Filter 9...\n");
        return -4;
    }

    //FILTER 10
    mpz_powm(aFQ, aF, phiQ, GRg);
    KAZ_DS_FILTER(aFQ, GRg, Q, hashValue, WB, W17);
    mpz_sub(W18, W17, S);
    mpz_mod(W18, W18, GRgQ);

    if(mpz_cmp_ui(W18, 0)==0){
        printf("Filter 10...\n");
        return -4;
    }

    //6) Verifying Procedures
    mpz_powm(tmp, R, S, Gg);
    mpz_powm(y1, g, tmp, N);

    mpz_powm(tmp, V, phiQ, GRg);
    mpz_mul(tmp, tmp, hashValue);
    mpz_mod(tmp, tmp, GRg);
    mpz_powm(tmp, R, tmp, Gg);
    mpz_powm(y2, g, tmp, N);

    if(mpz_cmp(y1, y2)!=0)
        return -4;

    memcpy(m, MBYTE, len);
    *mlen=len;

    mpz_clears(N, g, Gg, R, GRg, q, Q, phiQ, GRgqQ, hashValue, V, WA, WB, S, saltValue, NULL);
    mpz_clears(GRgQ, phiGRgQ, GhGRgQ, GVGRgQ, aF, W0, W1, W2, W3, W4, W5, W6, W7, W8, W9, W10, NULL);
    mpz_clears(W11, W12, W13, W14, W15, W16, W17, W18, VQ, aFQ, y1, y2, rnd, tmp, NULL);

	return 0;
}
